function [BF] = BF_Open(data,shift)
%% Code by BJD. Edited for readability 5/5/2018. For questions/clarification, please email at benjamin-decorte@uiowa.edu
% Function for computing a bayes factor testing the null hypothesis that no change
% is present in the data to the alternative hypothesis that the data were drawn from a
% shifted distribution. This shifted distribution is empirically-based using values in the 'shift' variable. 
% If such data are unavailable, you should use a flat/incremental
% alternative-prior analysis (available via Randy Gallistel's website: http://cognitivegenetic.rutgers.edu/ptn/). 
%
% Inputs:
% 1. data: data of interest that you wish to test evidence in favor of the
% null against that in favor of the alternative hypothesis
% 2. shift: data used to compute the alternative prior (i.e., the expected
% degree and direction of a shift in the 'data' vector under the alt hypothesis that a shift 
% did indeed occur). Should be composed of values from another 
% group or session (ideally, that in which a shift was observed; otherwise, the null and 
% alternative hypotheses may be equivalent and this test won't be useful).
%
% Outputs:
% BF: evidence in favor of the null hypothesis 

%% basic parameters for the different distributions. 
lower = -1; upper = 1; % limits on what values to evaluate when computing the likelihood functions (see below). should adjust this based on how large of a shift you see in your data. 
numSteps = 199; % n-1 bins for your data
delta = (upper-lower)/numSteps;
x = lower:delta:upper; % x-values for L(theta | d) under different hypotheses/the data of interest. 

%% calculate source standard deviation (SD)
% Estimate standard deviation from data and shift variables (weighted based on sample sizes in
% each data source)
Std = (length(shift)*std(shift)+length(data)*std(data))/(length(shift)+length(data)); % uses both data sets to compute SDs for likelihood functions
%% calculate likelihood function for the 'data' (hence, D in variable names) of interest
for i = 1:length(data)
    LLD(i,:) = log10(normpdf(data(i),x,Std)); % get individual log likelihood functions for each data point's potential distribution
end
LLDF = sum(LLD,1); % basically computing a posterior from individual likelihood functions. When re-linearized, should get a normal distribution, with a mean that's the average of all the data points (so long as equivalent SDs) and SD that is smaller than the estimated source SD (some rare exceptions are possible though). 
LDF = 10.^LLDF; % revert to linear scale
LDF = LDF/sum(LDF); % get area to equal 1. normalizing by sum

%% do same for distribution centered over 0 ('no-change' or null hypothesis, hence 'N' in variable names)
null = data-mean(data); % linearly shifted distribution from data of interest, centered on 0. Used for evidence that no change occurred. 
for i = 1:length(null)
    LLN(i,:) = log10(normpdf(null(i),x,Std)); 
end
LLNF = sum(LLN,1); 
LNF = 10.^LLNF;
LNF = LNF/sum(LNF); 

%% same to get alternative prior (a shift has occurred, with the degree of the shift being based on a specific dataset)
for i = 1:length(shift)
    LLA(i,:) = log10(normpdf(shift(i),x,Std)); 
end
LLAF = sum(LLA,1); 
LAF = 10.^LLAF; 
LAF = LAF/sum(LAF); 

%% get posteriors under different hypotheses. multiply element-wise.  
% null versus data
nullP = LNF.*LDF;
% alternative versus data
altP = LAF.*LDF;

%% compute bayes factor. 
BF = trapz(nullP)/trapz(altP); % Bayes factor: ratio of area under posteriors for the null hypothesis and alternative hypothesis. 
                               % when null is in the numerator, BF reflects evidence in favor of the null. 

%% good to plot and make sure all operations were successful. Also, helps you visualize why one hypothesis is being favored over another (or why neither is given strong support). 
% figure; hold on; plot(x,LDF); plot(x,LNF); plot(x,LAF,'k--')
% legend('data','null','alternative')
% title(['bayes factor = ' num2str(BF)])

end



